/*   Copyright (c) 1997      Cyril A. Vechera      St.-Petersburg, RUSSIA
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. The name of the author may not be used to endorse or promote products
 *     derived from this software without specific prior written permission.
 *  4. Any form of use of source and binaries are permitted only for
 *     noncommercial purpose. Any forms of commercial usage require
 *     direct author's permission.
 */

/*			RADIUS-client library
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <sys/time.h>

#include "radius.h"
#include "libradius.h"

/*************************************************************************/

/* macro definitions for construction your own RADIUS packets */
/* usially, requires two arguments - ptr (the place where we should insert
   atribute and value. Note, that ptr must be a lvalue */

/* User Name */
#define RADINS_USER_NAME(ptr, value) \
	*ptr++ = PW_USER_NAME; \
	*ptr++ = strlen(value) + 2; \
	strcpy (ptr, value); \
	ptr += strlen(value);

/* Calling-Station-ID */
#define RADINS_CALLING_STATION_ID(ptr, value) \
	*ptr++ = PW_CALLING_STATION_ID; \
	*ptr++ = strlen(value) + 2; \
	strcpy (ptr, value); \
	ptr += strlen(value);

/* State / connection string */
#define RADINS_STATE(ptr, value) \
	*ptr++ = PW_STATE; \
	*ptr++ = strlen(value) + 2; \
	strcpy (ptr, value); \
	ptr += strlen(value);

/* NAS-IP-Address */
#define RADINS_NAS_IP_ADDRESS(ptr, value) \
	*ptr++ = PW_NAS_IP_ADDRESS; \
	*ptr++ = 6; \
	memcpy(ptr, value, 4); \
	ptr += 4;

/* NAS-Port */
#define RADINS_NAS_PORT(ptr, value) \
	*ptr++ = PW_NAS_PORT; \
	*ptr++ = 6; \
	*(u_long *)ptr = htonl (value); \
	ptr += 4;

/* Acct-Status-Type */
#define RADINS_ACCT_STATUS_TYPE(ptr, value) \
	*ptr++ = PW_ACCT_STATUS_TYPE; \
	*ptr++ = 6; \
	*(u_long *)ptr = htonl(value); \
	ptr += 4;

/* Acct-Session-Id */
#define RADINS_ACCT_SESSION_ID(ptr, value) \
	*ptr++ = PW_ACCT_SESSION_ID; \
	*ptr++ = 2+strlen(value); \
	strcpy (ptr, value); \
	ptr += strlen (value);

/* Acct-Session-Time */
#define RADINS_ACCT_SESSION_TIME(ptr, value) \
	*ptr++ = PW_ACCT_SESSION_TIME; \
	*ptr++ = 6; \
	*(u_long *)ptr = htonl(value); \
	ptr += 4;

/* Service-Type */
#define RADINS_SERVICE_TYPE(ptr, value) \
	*ptr++ = PW_SERVICE_TYPE; \
	*ptr++ = 6; \
	*(u_long *) ptr = htonl(value); \
	ptr += 4;

/* Framed-Protocol */
#define RADINS_FRAMED_PROTOCOL(ptr, value) \
	*ptr++ = PW_FRAMED_PROTOCOL; \
	*ptr++ = 6; \
	*(u_long *)ptr = htonl(value); \
	ptr += 4;

/* User Password */
#define RADINS_USER_PASSWORD(ptr, value, auth) \
	*ptr++ = PW_USER_PASSWORD; \
	*ptr++ = 18; \
	radius_encrypt_password(ptr, value, auth); \
	ptr += 16;



int			radius_retry_max = RADLIB_DEFAULT_RETRY_MAX;
char			radius_secret[16] = "\0";
struct sockaddr_in	radius_me = {0, PF_INET, 0, {0}, "\0\0\0\0\0\0\0\0"};
struct sockaddr_in	radius_server = {0, PF_INET, 0, {0}, "\0\0\0\0\0\0\0\0"};
struct sockaddr_in	radius_servac = {0, PF_INET, 0, {0}, "\0\0\0\0\0\0\0\0"};
struct timeval		radius_timeout = {RADLIB_DEFAULT_TIMEOUT, 0};
int			radius_id = 0;
time_t			radius_session_start_time;
char			radius_session_id[254];
int			radius_accounting = 0;
int			radius_debug = 0;

/* initialization of all variables needed for the RADIUS access routines.
   local ip-addres for source, address and port of a RADIUS-server,
   shared secret */

int radius_init (char *config_file)
{
	FILE * fil;
	char argnam[128];
	char argval[128];
	
	if ( (fil = fopen (config_file, "r")) == NULL)
		return (0);
	radius_server.sin_port = htons(RADLIB_DEFAULT_RADIUS_PORT);
	radius_servac.sin_port = htons(RADLIB_DEFAULT_RADACC_PORT);
	while ( !feof (fil) )
	{
		fscanf (fil, "%s %s\n", argnam, argval);
		if ( !strcasecmp (argnam, "secret"))
			strncpy (radius_secret, argval, 16);
		else
		if ( !strcasecmp (argnam, "retry_max"))
			radius_retry_max = atoi (argval);
		else
		if ( !strcasecmp (argnam, "timeout"))
			radius_timeout.tv_sec = atoi (argval);
		else
		if ( !strcasecmp (argnam, "my_address"))
			radius_me.sin_addr.s_addr = inet_addr (argval);
		else
		if ( !strcasecmp (argnam, "rs_address"))
			radius_server.sin_addr.s_addr = inet_addr (argval);
		else
		if ( !strcasecmp (argnam, "rs_port"))
			radius_server.sin_port = htons (atoi(argval));
		if ( !strcasecmp (argnam, "ra_address"))
		{
			radius_accounting = 1;
			radius_servac.sin_addr.s_addr = inet_addr (argval);
		}
		else
		if ( !strcasecmp (argnam, "ra_port"))
			radius_servac.sin_port = htons (atoi(argval));
	}

	radius_session_start_time = time (NULL);

/* session identifier  =  ip_addr + pid + time */
	sprintf(radius_session_id, "%s %u %s",
		 inet_ntoa (radius_me.sin_addr),
		 getpid(),
		 ctime(&radius_session_start_time));
	return (1);
}

/* sends the query to a RADIUS server, waits an answer and returns it */
int radius_send_query (u_char *packet, u_char *reply, struct sockaddr_in *whom)
{
#define ERROR(x) {printf(x); return(0);}

	int sock;
	fd_set fd_wait;
	int retries = 0;
	struct sockaddr from;
	int fromlen;

	if ((sock = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
		ERROR("Can't socket\n");

	if (bind (sock, (struct sockaddr *) &radius_me, sizeof(radius_me)))
		ERROR("can't bind\n");

	for (;;)
	{
		if (sendto ( sock, packet, ntohs(*((u_short *)(packet + 2))), 0,
			 (struct sockaddr *)whom,
			 sizeof(struct sockaddr_in)) < 0)
			ERROR("can't sendto\n");

		FD_ZERO (&fd_wait);
		FD_SET (sock, &fd_wait);
		if ( select (sock+1, &fd_wait, NULL, NULL, &radius_timeout) < 0)
			ERROR("can't select\n");
		if (FD_ISSET (sock, &fd_wait))
			break;
		if ( ++retries == radius_retry_max)
			return (0);
	}

	fromlen = sizeof (from);
	if (recvfrom(sock, reply, RADLIB_MAX_PACKET_SIZE, 0, &from, &fromlen) < 0)
		ERROR("bad packet recieved\n");

	return(1);
#undef ERROR
}

/* encrypt password */
void radius_encrypt_password(char *ptr, char *passwd, char *auth)
{
	int slen;
	char buf[32];
	int i;

	slen = strlen (radius_secret);
	strcpy (buf, radius_secret);
	memcpy (buf + slen, auth, 16);
	md5_calc (ptr, buf, slen + 16);
	for (i = 0; i < strlen (passwd); i++)
		ptr[i] ^= passwd[i];
}

/* create Authenicator */
void radius_create_auth (char *ptr)
{
	char md5buf[128];
	int slen = strlen (radius_secret);
	TRadiusHeader *hdr;

	hdr = (TRadiusHeader *)ptr;
	strcpy (ptr+ntohs(hdr->length), radius_secret);
	md5_calc (md5buf, ptr, slen + ntohs (hdr->length));
	memcpy (hdr->auth, md5buf, 16);
}

int radius_response_auth (char *resp, char *auth)
{
	u_char tmp[RADLIB_MAX_PACKET_SIZE];
	u_char md5buf[128];
	int len;
	
	len = ntohs (*(u_short*)(resp+2));
	memcpy (tmp, resp, len);
	memcpy (tmp + 4, auth, 16);
	memcpy (tmp + len, radius_secret, strlen(radius_secret));

	md5_calc (md5buf, tmp, len + strlen(radius_secret));
	return (!memcmp (md5buf, resp + 4, 16));
}

int radius_account_start (u_char *user_name, u_long port)
{
	u_char reply[RADLIB_MAX_PACKET_SIZE];
	u_char request[RADLIB_MAX_PACKET_SIZE];
	TRadiusHeader *hdr;
	TRadiusHeader *rep;
	u_char *ptr;
	int i;

	hdr = (TRadiusHeader *)request;
	hdr->code = PW_ACCOUNTING_REQUEST;
	hdr->id = radius_id++;
	for (i = 0; i < 4; i++)
		hdr->auth[i] = 0L;
	ptr = request + 20;
	RADINS_USER_NAME (ptr, user_name);
	RADINS_NAS_IP_ADDRESS (ptr, &radius_me.sin_addr.s_addr);
	RADINS_NAS_PORT (ptr, port);
	RADINS_CALLING_STATION_ID (ptr, callerid);
	RADINS_STATE (ptr, con_state);
	RADINS_ACCT_STATUS_TYPE (ptr, PW_STATUS_START);
	RADINS_ACCT_SESSION_ID (ptr, radius_session_id);
	hdr->length = htons ((unsigned)ptr - (unsigned)request);
	radius_create_auth (request);

/* Ok, packet is now assembled. send it */
	if (!radius_send_query (request, reply, &radius_servac))
		return(RADLIB_NO_RESPONSE);
/* verify */
	rep = (TRadiusHeader *)reply;
	if (rep->id != hdr->id)
		return(RADLIB_BAD_ID);
	if (!radius_response_auth (reply, (char *)hdr->auth))
		return(RADLIB_BAD_AUTH);
	if (rep->code != PW_ACCOUNTING_RESPONSE)
		return(RADLIB_ACCESS_DENIED);
	return (0); /* all right */
}

int radius_account_stop (u_char *user_name, u_long port, time_t stime)
{
	u_char reply[RADLIB_MAX_PACKET_SIZE];
	u_char request[RADLIB_MAX_PACKET_SIZE];
	TRadiusHeader *hdr;
	TRadiusHeader *rep;
	u_char *ptr;
	int i;

	hdr = (TRadiusHeader *)request;
	hdr->code = PW_ACCOUNTING_REQUEST;
	hdr->id = radius_id++;
	for (i = 0; i < 4; i++)
		hdr->auth[i] = 0L;
	ptr = request + 20;
	RADINS_USER_NAME (ptr, user_name);
	RADINS_NAS_IP_ADDRESS (ptr, &radius_me.sin_addr.s_addr);
	RADINS_NAS_PORT (ptr, port);
	RADINS_CALLING_STATION_ID (ptr, callerid);
	RADINS_STATE (ptr, con_state);
	RADINS_ACCT_STATUS_TYPE (ptr, PW_STATUS_STOP);
	RADINS_ACCT_SESSION_ID (ptr, radius_session_id);
	RADINS_ACCT_SESSION_TIME (ptr, stime);
	hdr->length = htons ((int)ptr - (int)request);
	radius_create_auth (request);

	if (!radius_send_query (request, reply, &radius_servac))
		return(RADLIB_NO_RESPONSE);
/* verify */
	rep = (TRadiusHeader *)reply;
	if (rep->id != hdr->id)
		return(RADLIB_BAD_ID);
	if (!radius_response_auth (reply, (char *)hdr->auth))
		return(RADLIB_BAD_AUTH);
	if (rep->code != PW_ACCOUNTING_RESPONSE)
		return(RADLIB_ACCESS_DENIED);
	return (0); /* all right */
}

/* returns client's ip-addres, netmask, MTU, Compression type */
int radius_request_ppp_PAP (char *user_name, char *user_passwd, u_long port,
	u_long	**address, u_long **netmask, u_long **mtu, u_long **compress)
{
	u_char reply[RADLIB_MAX_PACKET_SIZE];
	u_char request[RADLIB_MAX_PACKET_SIZE];
	TRadiusHeader *hdr;
	TRadiusHeader *rep;
	u_char *ptr;
	int i;

	int is_address = 0;
	int is_netmask = 0;
	int is_mtu = 0;
	int is_compress = 0;

	hdr = (TRadiusHeader *)request;
	hdr->code = PW_ACCESS_REQUEST;
	hdr->id = radius_id++;
	srandom (time (NULL));
	for (i = 0; i < 4; i++)
		hdr->auth[i] = random();
	ptr = request + 20;
	RADINS_USER_NAME(ptr, user_name);
	RADINS_USER_PASSWORD(ptr, user_passwd, hdr->auth);
	RADINS_SERVICE_TYPE (ptr, PW_FRAMED);
	RADINS_FRAMED_PROTOCOL (ptr, PW_PPP);
	RADINS_NAS_IP_ADDRESS(ptr, &radius_me.sin_addr.s_addr);
	RADINS_NAS_PORT (ptr, port);
	RADINS_CALLING_STATION_ID (ptr, callerid);
	RADINS_STATE (ptr, con_state);
	hdr->length = htons ((int)ptr - (int)request);
	if (!radius_send_query (request, reply, &radius_server))
		return(RADLIB_NO_RESPONSE);
/* verify */
	rep = (TRadiusHeader *)reply;
	if (rep->id != hdr->id)
		return(RADLIB_BAD_ID);
	if (!radius_response_auth (reply, (char *)hdr->auth))
		return(RADLIB_BAD_AUTH);
	if (rep->code == PW_ACCESS_REJECT)
		return(RADLIB_ACCESS_DENIED);
	if (rep->code != PW_ACCESS_ACCEPT)
		return(RADLIB_BAD_CODE);
	for  (  ptr = reply + sizeof(TRadiusHeader);
		ptr < reply + ntohs(rep->length);
		ptr += *(ptr+1))
	{
		switch (*ptr)
		{
			case PW_FRAMED_IP_ADDRESS:
				is_address = 1;
				memcpy (*address, ptr + 2, 4);
				break;
			case PW_FRAMED_IP_NETMASK:
				is_netmask = 1;
				memcpy (*netmask, ptr + 2, 4);
				break;
			case PW_FRAMED_MTU:
				is_mtu = 1;
				**mtu = ntohl(*(u_long*)(ptr + 2));
				break;
			case PW_FRAMED_COMPRESSION:
				is_compress = 1;
				**compress = ntohl(*(u_long*)(ptr + 2));
				break;
		}			

	}
	if (!is_address) *address = NULL;
	if (!is_netmask) *netmask = NULL;
	if (!is_mtu) *mtu = NULL;
	if (!is_compress) *compress = NULL;
	return (0); /* all right */
}
