/**	Создание файлов в каталоге mykernel_test в файловой системе /proc/
	mykernel_test/jiffies - количество тиков таймера
	mykernel_test/jiffies2 - ссылка на mykernel_test/jiffies
	mykernel_test/seconds - количество секунд работы системы
	mykernel_test/foo - печатает сообщения
	mykernel_test/success - поздравления	**/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/jiffies.h>
#include <asm/uaccess.h>

#define MYKERNEL_ENTRY "mykernel_test"

#define FOO_LEN 16

struct fb_data_t {
	char value[FOO_LEN + 1];
};

static struct proc_dir_entry 
	*example_dir, *jiffies_file, *symlink, *seconds_file, *foo_file, *success_file;

struct fb_data_t foo_data, bar_data;

static int proc_read_jiffies
	(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len;

	len = sprintf(page, "Your processor timer ticked %lu times\n", jiffies);

	return len;
}

static int proc_read_seconds
	(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	unsigned long sec;

	sec = jiffies / HZ;

	len = sprintf(page, "Your Linux system works %lu seconds\n", sec);

	return len;
}

static int proc_read_foo
	(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	struct fb_data_t *fb_data = (struct fb_data_t *)data;

	len = sprintf(page, "Last message: %s", fb_data->value);

	return len;
}

static int proc_write_foo
	(struct file *file, const char *buffer, unsigned long count, void *data)
{
	int len;
	struct fb_data_t *fb_data = (struct fb_data_t *)data;

	if(count > FOO_LEN)
		len = FOO_LEN;
	else
		len = count;

	if(copy_from_user(fb_data->value, buffer, len))
		return -EFAULT;

	fb_data->value[len] = '\0';

	return len;
}

static int proc_read_success
	(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len;

	len = sprintf(page, "Congratulations!\nYour Linux experiment successed!\n");

	return len;
}

static int __init procfiles_init(void)
{
	int rv = 0;

	///создание каталога /proc/mykernel_test
	example_dir = proc_mkdir(MYKERNEL_ENTRY, NULL);
	if (example_dir == NULL) {
		rv = -ENOMEM;
		goto out;
	}
	example_dir->owner = THIS_MODULE;
	
	///создание файла jiffies
	jiffies_file = 
		create_proc_read_entry("jiffies", 0444, example_dir, proc_read_jiffies, NULL);
	if (jiffies_file == NULL) {
		rv  = -ENOMEM;
		goto no_jiffies;
	}
	jiffies_file->owner = THIS_MODULE;

	///создание симлинка на jiffies
	symlink = proc_symlink("jiffies2", example_dir, "jiffies");
	if (symlink == NULL) {
		rv = -ENOMEM;
		goto no_symlink;
	}
	symlink->owner = THIS_MODULE;

	///создание файла seconds
	seconds_file = 
		create_proc_read_entry("seconds", 0444, example_dir, proc_read_seconds, NULL);
	if (seconds_file == NULL) {
		rv  = -ENOMEM;
		goto no_seconds;
	}
	seconds_file->owner = THIS_MODULE;

	///создание файлов foo
	foo_file = create_proc_entry("foo", 0644, example_dir);
	if (foo_file == NULL) {
		rv = -ENOMEM;
		goto no_foo;
	}
	strcpy(foo_data.value, "foo\n");
	foo_file->data = &foo_data;
	foo_file->read_proc = proc_read_foo;
	foo_file->write_proc = proc_write_foo;
	foo_file->owner = THIS_MODULE;

	///создание файла success
	success_file = 
		create_proc_read_entry("success", 0444, example_dir, proc_read_success, NULL);
	if (success_file == NULL) {
		rv = -ENOMEM;
		goto no_success;
	}
	success_file->owner = THIS_MODULE;

	printk(KERN_INFO "/proc/%s created\n", MYKERNEL_ENTRY);
	return 0;

	no_success:
		remove_proc_entry("foo", example_dir);
	no_foo:
		remove_proc_entry("seconds", example_dir);
	no_seconds:
		remove_proc_entry("jiffies2", example_dir);
	no_symlink:
		remove_proc_entry("jiffies", example_dir);
	no_jiffies:
		remove_proc_entry(MYKERNEL_ENTRY, NULL);
	out:
		return rv;
}


static void __exit procfiles_exit(void)
{
	remove_proc_entry("success", example_dir);
	remove_proc_entry("foo", example_dir);
	remove_proc_entry("seconds", example_dir);
	remove_proc_entry("jiffies2", example_dir);
	remove_proc_entry("jiffies", example_dir);
	remove_proc_entry(MYKERNEL_ENTRY, NULL);

	printk(KERN_INFO "/proc/%s removed\n", MYKERNEL_ENTRY);
}


module_init(procfiles_init);
module_exit(procfiles_exit);

MODULE_LICENSE("GPL");
