/**	Для создания новой файловой системы требуется
	перекрыть существующие функции структур VFS.	**/
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/pagemap.h>
#include <linux/fs.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>

#define LFS_MAGIC 0x19980122
static atomic_t counter, subcounter;

/**	Для представления файлов или каталогов, нам необходимо создать inode.
	Функция делает именно это.	**/
static struct inode *linfs_make_inode(struct super_block *sb, int mode)
{
	struct inode *ret = new_inode(sb);
	if (ret) {
		ret->i_mode = mode;
		ret->i_uid = ret->i_gid = 0;
		ret->i_blksize = PAGE_CACHE_SIZE;
		ret->i_blocks = 0;
		ret->i_atime = ret->i_mtime = ret->i_ctime = CURRENT_TIME;
	}
	return ret;
}

/**	Открытие файла.
	Для этого скопируем указатель на файл.	**/
static int linfs_open(struct inode *inode, struct file *filp)
{
	filp->private_data = inode->u.generic_ip;
	return 0;
}

#define TMPSIZE 20
/**	Чтение файла.
	Здесь увеличивается значение счетчика числа обращений к файлу.
	Значение счетчика увеличивается если мы читаем с начала файла (offset = 0).	**/
static ssize_t linfs_read_file
	(struct file *filp, char *buf, size_t count, loff_t *offset)
{
	atomic_t *counter = (atomic_t *) filp->private_data;
	int v, len;
	char tmp[TMPSIZE];
	///Прочитать текущее значение
	v = atomic_read(counter);
	if (*offset == 0)
		atomic_inc(counter);
	else
		v -= 1;
	len = snprintf(tmp, TMPSIZE, "%d\n", v);
	if (*offset > len)
		return 0;
	if (count > len - *offset)
		count = len - *offset;
	///Вернуть значение пользователю, увеличить смещение
	if (copy_to_user(buf, tmp + *offset, count))
		return -EFAULT;
	*offset += count;
	return count;
}

/**	Запись в файл.	**/
static ssize_t linfs_write_file
	(struct file *filp, const char *buf, size_t count, loff_t *offset)
{
	atomic_t *counter = (atomic_t *) filp->private_data;
	char tmp[TMPSIZE];
	///Записывать только с начала
	if (*offset != 0)
		return -EINVAL;
	///Прочитать значение от пользователя
	if (count >= TMPSIZE)
		return -EINVAL;
	memset(tmp, 0, TMPSIZE);
	if (copy_from_user(tmp, buf, count))
		return -EFAULT;
	///Сохранить это значение
	atomic_set(counter, simple_strtol(tmp, NULL, 10));
	return count;
}

/**	Структура file_operations - операции с файлами.	**/
static struct file_operations linfs_fops = {
	.open	= linfs_open,
	.read 	= linfs_read_file,
	.write  = linfs_write_file,
};

/**	Создание файла.	**/
static struct dentry *linfs_create_file
	(struct super_block *sb, struct dentry *dir, const char *name, atomic_t *counter)
{
	struct dentry *dentry;
	struct inode *inode;
	struct qstr qname;
	
	///Создать хэш-версию имени файла
	qname.name = name;
	qname.len = strlen(name);
	qname.hash = full_name_hash(name, qname.len);
	///Создаем dentry и inode для работы
	dentry = d_alloc(dir, &qname);
	if (!dentry)
		goto out;
	inode = linfs_make_inode(sb, S_IFREG | 0644);
	if (!inode)
		goto out_dput;
	inode->i_fop = &linfs_fops;
	inode->u.generic_ip = counter;
	///Поместить в dentry cache
	d_add(dentry, inode);
	return dentry;
	///На случай ошибок
	out_dput:
		dput(dentry);
	out:
		return 0;
}

/**	Создание каталога очень похоже на создание файла.
	Но каталоги в Unix - это тоже файлы.	**/
static struct dentry *linfs_create_dir
	(struct super_block *sb, struct dentry *parent, const char *name)
{
	struct dentry *dentry;
	struct inode *inode;
	struct qstr qname;
	
	qname.name = name;
	qname.len = strlen (name);
	qname.hash = full_name_hash(name, qname.len);
	dentry = d_alloc(parent, &qname);
	if (! dentry)
		goto out;
	
	inode = linfs_make_inode(sb, S_IFDIR | 0644);
	if (! inode)
		goto out_dput;
	inode->i_op = &simple_dir_inode_operations;
	inode->i_fop = &simple_dir_operations;
	
	d_add(dentry, inode);
	return dentry;
	
	out_dput:
		dput(dentry);
	out:
		return 0;
}

/**	Создние первых файлов.	**/
static void linfs_create_files(struct super_block *sb, struct dentry *root)
{
	struct dentry *subdir;
	///Файл counter
	atomic_set(&counter, 0);
	linfs_create_file(sb, root, "counter", &counter);
	///Файл subdir/subcounter
	atomic_set(&subcounter, 0);
	subdir = linfs_create_dir(sb, root, "subdir");
	if (subdir)
		linfs_create_file(sb, subdir, "subcounter", &subcounter);
}

/**	Структура super_operations.	**/
static struct super_operations linfs_sops = {
	.statfs		= simple_statfs,
	.drop_inode	= generic_delete_inode,
};

/**	Заполнение суперблока при создании инициализации ФС.	**/
static int linfs_fill_super(struct super_block *sb, void *data, int silent)
{
	struct inode *root;
	struct dentry *root_dentry;
	///Основные параметры
	sb->s_blocksize = PAGE_CACHE_SIZE;
	sb->s_blocksize_bits = PAGE_CACHE_SHIFT;
	sb->s_magic = LFS_MAGIC;
	sb->s_op = &linfs_sops;
	///Создадим inode в ядре для представления корневого каталога ФС
	root = linfs_make_inode (sb, S_IFDIR | 0755);
	if (!root)
		goto out;
	root->i_op = &simple_dir_inode_operations;
	root->i_fop = &simple_dir_operations;
	///Заставим dentry отображаться в ядре
	root_dentry = d_alloc_root(root);
	if (!root_dentry)
		goto out_iput;
	sb->s_root = root_dentry;
	///Создание первых файлов
	linfs_create_files (sb, root_dentry);
	return 0;
	
	out_iput:
		iput(root);
	out:
		return -ENOMEM;
}

/**	Необходимо для регистрации ФС.	**/
static struct super_block *linfs_get_super
	(struct file_system_type *fst, int flags, char *devname, void *data, struct vfsmount *mnt)
{
	return get_sb_single(fst, flags, data, linfs_fill_super, mnt);
}

/**	Файловая система.	**/
static struct file_system_type linfs_type = {
	.owner 		= THIS_MODULE,
	.name		= "linfs",
	.get_sb		= linfs_get_super,
	.kill_sb	= kill_litter_super,
};

/**	Точки входа и выхода модуля.	**/
static int __init linfs_init(void)
{
	return register_filesystem(&linfs_type);
}

static void __exit linfs_exit(void)
{
	unregister_filesystem(&linfs_type);
}

module_init(linfs_init);
module_exit(linfs_exit);

MODULE_LICENSE("GPL");
